﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Hims.Api.Utilities;
    using Hims.Domain.Services;
    using Hims.Shared.DataFilters;
    using Hims.Shared.EntityModels;
    using Hims.Shared.UserModels.Filters;
    using Hims.Api.Models;

    // ReSharper disable StyleCop.SA1126

    /// <summary>
    /// The patients controller.
    /// </summary>
    [Route("api/auditlog")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class AuditLogController : BaseController
    {
        /// <summary>
        /// the audit log service
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public AuditLogController(IAuditLogService auditLogServices) => this.auditLogServices = auditLogServices;

        /// <summary>
        /// To fetch audit logs
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The list of audit logs.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of auditLogs.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("fetch")]
        [ProducesResponseType(typeof(IEnumerable<AuditLogModel>), 200)]
        [ProducesResponseType(500)]

        public async Task<ActionResult> FetchAsync([FromBody] AuditLogFilterModel model, [FromHeader] LocationHeader header)
        {
            model = (AuditLogFilterModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var auditLogs = await auditLogServices.FetchAsync(model);
            if (auditLogs == null)
            {
                return ServerError();
            }

            return Success(auditLogs);
        }

        /// <summary>
        /// To insert audit logs
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <param name="header"></param>
        /// <returns>
        /// The list of audit logs.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of auditLogs.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("insert")]
        [ProducesResponseType(500)]
        public async Task<int> InsertAsync([FromBody] AuditLogModel model,[FromHeader] LocationHeader header)
        {
            model = (AuditLogModel)EmptyFilter.Handler(model);
            model.LocationId = header != null && !string.IsNullOrEmpty(header.LocationId) ? Convert.ToInt32(header.LocationId) : (int?)null;
            model.LogDate = DateTime.UtcNow;
            await auditLogServices.LogAsync(model);
            return 1;
        }
    }
}
